<?php
/*
Plugin Name: WooCommerce ParcelPanel integration
Description: Transfer tracking number from note to ParcelPanel
Version: 1.3
Author: Denis Svishchev
*/

if (!defined('ABSPATH')) {
    exit;
}

// Add PARCELPANEL_API_KEY to wp-config before enable the plugin
if (!defined('PARCELPANEL_API_KEY')) {
    error_log('PARCELPANEL_API_KEY not set in .env file');
    define('PARCELPANEL_API_KEY', '');
}

/*
 * courier_code => preg_pattern
 * Huboo dispatched order – Tracking: LY525357123DE Url: https://packet.deutschepost.com/web/portal-europe/packet_traceit?barcode=LY525357123DE
 * Huboo dispatched order – Tracking: H01HYA0038323152 Url: https://www.evri.com/track/parcel/H01HYA0038323152/details
 * Huboo dispatched order – Tracking: QD574723248GB Url: https://www.royalmail.com/track-your-item#/tracking-results/QD574723248GB
 * */
const TRACKING_NUM_PREFIX = '/Tracking:\s([A-Z0-9]+)\s';
const COURIER_PATTERNS = [
    'deutsche-post' => TRACKING_NUM_PREFIX . '.+deutschepost\.com/',
    'hermes-uk' => TRACKING_NUM_PREFIX . '.+evri\.com/',
    'royal-mail' => TRACKING_NUM_PREFIX . '.+royalmail\.com/',
    /* new */
    'postnl-3s' => TRACKING_NUM_PREFIX . '.+postnl\.nl/',
    'fedex' => TRACKING_NUM_PREFIX . '.+fedex\.com/',
    'dpd' => TRACKING_NUM_PREFIX . '.+dpd\.de/',
    'gls' => TRACKING_NUM_PREFIX . '.+gls-group\.eu/',
];


// Transfer tracking number and other info to ParcelPanel (creating parcelpanel shipment in the WC->Orders->Order section)
function transfer_tracking_number_from_note_to_parcelpanel ($note_id, $order) {

    $order_id = $order->get_id();
    $note_std = wc_get_order_note($note_id);
    if(!$note_std) {
        return;
    }

    $tracking_data = get_tracking_number_from_order_note($note_std->content, COURIER_PATTERNS);
    if(!$tracking_data) {
        return;
    }

    // Check parcelpanel record before create new one.
//    $tracking_std = get_tracking_by_order_id($order_id);
//    if(!empty($tracking_std->data)) {
//        return;
//    }

    create_tracking($order_id, $tracking_data);
}
add_action('woocommerce_order_note_added', 'transfer_tracking_number_from_note_to_parcelpanel', 10, 2 );

function get_tracking_by_order_id(int $order_id): mixed {
    $url = 'https://wp-api.parcelpanel.com/api/v1/tracking/list';
    $payload = [
        'orders' => [
            [
                'order_id' => $order_id
            ]
        ]
    ];
    return post_to_parcelpanel($url, $payload);
}

function create_tracking(int $order_id, array $tracking_data) {
    $url = 'https://wp-api.parcelpanel.com/api/v1/tracking/create';
    $payload = [
        'shipments' => [
            [
                'order_id' => $order_id,
                'tracking_number' => $tracking_data['tracking_number'],
                'courier_code' => $tracking_data['courier_code'],
            ]
        ]
    ];

    return post_to_parcelpanel($url, $payload);
}

function post_to_parcelpanel(string $url, array $payload): mixed {

    $headers = [
        'Content-Type: application/json',
        'PP-Api-Key: ' . PARCELPANEL_API_KEY,
    ];

    $curl = curl_init();

    curl_setopt_array($curl, array(
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => '',
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => 'POST',
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => $headers,
    ));

    $response = curl_exec($curl);
    $http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);

    if(!$response) {
        parcelpanel_log(curl_error($curl));
    }

    if ($http_status >= 200 && $http_status < 300) {
        return json_decode($response);
    }
    parcelpanel_log($response);
    return false;
}

function get_tracking_number_from_order_note(string $note, array $patterns): bool|array {
    foreach ($patterns as $courier_code => $pattern) {
        if (preg_match($pattern, $note, $matches)) {
            return [
                'tracking_number' => $matches[1],
                'courier_code' => $courier_code,
            ];
        }
    }
    return false;
}

function parcelpanel_log($mixture) {
    $upload_dir = wp_upload_dir();
    $log_message = '[' . date('Y-m-d H:i:s') . '] ' . print_r($mixture, true) . PHP_EOL;
    $log_file = $upload_dir['basedir'].'/parcelpanel.log';
    file_put_contents($log_file, $log_message, FILE_APPEND);
}