<?php

declare(strict_types=1);

namespace Affilyads\WoocommerceKafkaIntegration\Includes;

use Affilyads\WoocommerceKafkaIntegration\Includes\Connectors\ClassWoocommerceKafkaConnector;
use Affilyads\WoocommerceKafkaIntegration\Includes\Connectors\ClassWoocommerceKafkaRawConnector;
use Affilyads\WoocommerceKafkaIntegration\Includes\Topics\ClassWoocommerceKafkaOrderCreatedTopic;
use Affilyads\WoocommerceKafkaIntegration\Includes\Topics\ClassWoocommerceKafkaOrderUpdatedTopic;

final class ClassWoocommerceKafkaCore
{

	protected static ?ClassWoocommerceKafkaCore $instance = null;

	protected ClassWoocommerceKafkaConnector $connector;
	protected ?ClassWoocommerceKafkaRawConnector $rawConnector;

	public static function getInstance(): self
	{
		if (null === self::$instance) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	public function __construct()
	{
		$this->connector = ClassWoocommerceKafkaConnector::getInstance();

		$this->rawConnector = ClassWoocommerceKafkaRawConnector::getInstance();

		if (!$this->rawConnector->isCheckedDependencies()) {
			$this->rawConnector = null;
		}


		if ($this->connector->isCheckedDependencies() || $this->rawConnector) {
			$this->addEventListeners();
			$this->registerScheduleActions();
		}
	}

	/**
	 * @param int $order_id
	 * @param mixed $order
	 *
	 * @return void
	 */
	public function handleWoocommerceOrderUpdated(int $order_id, mixed $order): void
	{
		$cache_key = "order_update_$order_id";

		error_log("handleWoocommerceOrderUpdated $cache_key");

		wp_cache_delete($cache_key);

		as_unschedule_action("send_kafka_order_update", [$order_id]);

		as_schedule_single_action(time() + 5, "send_kafka_order_update", [$order_id]);
	}

	/**
	 * @param int $order_id
	 * @param mixed $order
	 *
	 * @return void
	 */
	public function handleWoocommerceOrderCreated(int $order_id, mixed $order): void
	{
		$cache_key = "order_create_$order_id";

		error_log("handleWoocommerceOrderCreated $cache_key");

		wp_cache_delete($cache_key);

		as_unschedule_action("send_kafka_order_create", [$order_id]);

		as_schedule_single_action(time() + 5, "send_kafka_order_create", [$order_id]);
	}

	private function registerScheduleActions(): void
	{
		add_action( 'send_kafka_order_create', [
			new ClassWoocommerceKafkaOrderCreatedTopic($this->connector, $this->rawConnector),
			'send'
		], 10, 2 );

		add_action( 'send_kafka_order_update', [
			new ClassWoocommerceKafkaOrderUpdatedTopic($this->connector, $this->rawConnector),
			'send'
		], 10, 2 );
	}

	private function addEventListeners(): void
	{
		add_action( 'woocommerce_new_order', [$this, 'handleWoocommerceOrderCreated'], 10, 2 );

		add_action('woocommerce_update_order', [$this, 'handleWoocommerceOrderUpdated'], 10, 2 );
		add_action('woocommerce_order_refunded', [$this, 'handleWoocommerceOrderUpdated'], 10, 2 );
	}

}