<?php

declare( strict_types=1 );

namespace Affilyads\WoocommerceKafkaIntegration\Includes\Topics;

use Affilyads\WoocommerceKafkaIntegration\Includes\Connectors\ClassWoocommerceKafkaConnector;
use Affilyads\WoocommerceKafkaIntegration\Includes\Connectors\ClassWoocommerceKafkaRawConnector;
use GuzzleHttp\Exception\GuzzleException;
use KafkaIntegration\Vendor\Opis\JsonSchema\Errors\ErrorFormatter;
use KafkaIntegration\Vendor\Opis\JsonSchema\ValidationResult;
use KafkaIntegration\Vendor\Opis\JsonSchema\Validator;
use RdKafka\ProducerTopic;

abstract class ClassWoocommerceKafkaAbstractTopic {

	const DATE_KEYS = [
		'date_created',
		'date_created_gmt',
		'date_modified',
		'date_modified_gmt',
		'date_completed',
		'date_completed_gmt',
		'date_paid',
		'date_paid_gmt'
	];

	private ClassWoocommerceKafkaConnector $connector;
	private ?ClassWoocommerceKafkaRawConnector $rawConnector;
	private array $topics = [];
	private ?ProducerTopic $rawTopic = null;
	private ?ProducerTopic $dlqTopic = null;

	abstract protected function getTopicName(): string;

	abstract protected function prepareData(): array;

	abstract public function send(): void;

	protected function getRawTopicName(): string {
		if ( defined( 'WP_KAFKA_RAW_ORDERS_TOPIC_NAME' ) && null !== constant( 'WP_KAFKA_RAW_ORDERS_TOPIC_NAME' ) ) {
			return WP_KAFKA_RAW_ORDERS_TOPIC_NAME;
		}

		return 'orders_raw';
	}

	protected function getDlqTopicName(): string {
		if ( defined( 'WP_KAFKA_RAW_ORDERS_DLQ_TOPIC_NAME' ) && null !== constant( 'WP_KAFKA_RAW_ORDERS_DLQ_TOPIC_NAME' ) ) {
			return WP_KAFKA_RAW_ORDERS_DLQ_TOPIC_NAME;
		}

		return 'orders_dlq';
	}

	public function __construct( ClassWoocommerceKafkaConnector $connector, ?ClassWoocommerceKafkaRawConnector $rawConnector = null ) {
		$this->connector    = $connector;
		$this->rawConnector = $rawConnector;

		$this->initTopics();
		$this->initRawTopics();
	}

	private function initTopics(): void {
		if ( ! $this->connector->isCheckedDependencies() ) {
			return;
		}

		$topicNames = array_filter( array_map( 'trim', explode( ',', $this->getTopicName() ) ) );

		foreach ( $topicNames as $topicName ) {
			$this->topics[] = $this->connector->getProducer()->newTopic( $topicName );
		}
	}

	private function initRawTopics() {
		if ( ! $this->rawConnector || ! $this->rawConnector->isCheckedDependencies() ) {
			return;
		}

		error_log('Init rawTopics');

		$this->rawTopic = $this->rawConnector->getProducer()->newTopic( $this->getRawTopicName() );
		$this->dlqTopic = $this->rawConnector->getProducer()->newTopic( $this->getDlqTopicName() );
	}

	protected function sendData( array $preparedData ): void {
		if ( ! $this->connector->isCheckedDependencies() ) {
			return;
		}

		foreach ( $this->topics as $topic ) {
			$topic->produce( RD_KAFKA_PARTITION_UA, 0, json_encode( $preparedData ), (string) $preparedData['id'] ?? null );
		}

		$this->connector->getProducer()->flush( 10000 );
	}

	/**
	 * @param array $data Data to validate
	 * @param string $topic Topic to validate
	 * @param string $postfix Postfix to validate key/value
	 *
	 * @return bool
	 */
	protected function checkSchemaRegistry(array $data, string $topic, string $postfix = 'value'): ?ValidationResult {

		try {
			if (!wp_cache_get("schema_registry_client_$topic-$postfix")) {
				$schemaResponse = $this->rawConnector->getSchemaRegistryClient()->get(
					"http://206.189.3.204:9081/subjects/$topic-$postfix/versions/latest"
				);

				$jsonResponse = json_decode($schemaResponse->getBody()->getContents());

				wp_cache_set("schema_registry_client_$topic-$postfix", $jsonResponse->schema, '', DAY_IN_SECONDS);
			}

			$schema  = wp_cache_get("schema_registry_client_$topic-$postfix");

			$validator = new Validator();
			$validator->setMaxErrors(10);
			$validator->setStopAtFirstError(false);

			$stdData = json_decode(json_encode($data));

			return $validator->validate($stdData, $schema);
		} catch (GuzzleException $e) {
			error_log( $e->getMessage() );
		}

		return null;
	}

	protected function sendRawData( array $preparedData ): void {
		if ( ! $this->rawConnector || ! $this->rawConnector->isCheckedDependencies() ) {
			return;
		}

		foreach ( self::DATE_KEYS as $key ) {
			if ( null === $preparedData[$key] ) {
				continue;
			}

			if ( "" === $preparedData[$key] ) {
				$preparedData[$key] = null;
				continue;
			}

			$preparedData[$key] .= 'Z';
		}

		if (count($preparedData['line_items']) > 0) {
			$preparedData['line_items'][0]['price'] = sprintf("%.2f", $preparedData['line_items'][0]['price']);
		}

		$rawTopicName = $this->getRawTopicName();

		$body = [
			'key'   => [
				'data' => [
					'id'   => $preparedData['id'] ?? null,
					'code' => WP_KAFKA_RAW_CHANNEL_CODE
				]
			],
			'value' => [
				'data' => $preparedData
			]
		];

		$validationValueResult = $this->checkSchemaRegistry($body['value']['data'], $rawTopicName);
		$validationKeyResult = $this->checkSchemaRegistry($body['key']['data'], $rawTopicName, 'key');

		if (
			($validationValueResult && $validationValueResult->isValid()) &&
			($validationKeyResult && $validationKeyResult->isValid())
		) {
			$this->rawTopic->produce(
				RD_KAFKA_PARTITION_UA,
				0,
				json_encode($body['value']['data']),
				json_encode($body['key']['data'])
			);

			$this->rawConnector->getProducer()->flush( 10000 );
		} else {
			$valueErrors = $validationValueResult->error() ? (new ErrorFormatter())->format($validationValueResult->error()) : [];
			$keyErrors = $validationKeyResult->error() ? (new ErrorFormatter())->format($validationKeyResult->error()) : [];

			$errors = [];

			foreach (array_merge($valueErrors, $keyErrors) as $field => $error) {
				$errors[] = $field . ': ' . implode(', ', $error);
			}

			$concatErrors = implode(',', $errors);

			$this->sendDataToDlq( $body, $concatErrors );
		}
	}

	protected function sendDataToDlq(array $body, string $reason): void {
		error_log('sendDataToDlq called');

		$body['headers'] = [
			'dlq_reason' => $reason,
			'dlq_timestamp' => (string) time(),
			'dlq_attempt_counter' => 1
		];

		$this->dlqTopic->producev(
			RD_KAFKA_PARTITION_UA,
			0,
			json_encode($body['value']['data']),
			json_encode($body['key']['data']),
			$body['headers']
		);

		$this->rawConnector->getProducer()->flush( 10000 );
	}

}