<?php

declare(strict_types=1);

namespace woocommerce_kafka_integration\includes\front;

use woocommerce_kafka_integration\includes\Class_Woocommerce_Kafka_Historical;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Class_Woocommerce_Kafka_ACF_Page {

    protected const NOTICE_OPTION = 'kafka_historical_admin_notice';

    public static function init(): void {
        if ( ! function_exists( 'acf_add_options_page' ) ) {
            return;
        }

        add_action( 'acf/init', [ __CLASS__, 'register_options_page' ] );

        add_action( 'acf/save_post', [ __CLASS__, 'save_options' ], 20 );

        add_action( 'admin_notices', [ __CLASS__, 'admin_notice' ], 99 );

    }

    public static function register_options_page(): void {
        acf_add_options_page( [
            'page_title'  => self::get_translation('Kafka Historical'),
            'menu_title'  => self::get_translation( 'Kafka Historical' ),
            'capability'  => 'manage_woocommerce',
            'parent_slug' => 'tools.php',
            'updated_message' => '',
        ] );

        acf_add_local_field_group([
            'key'                   => 'group_kafka_historical',
            'title'                 => self::get_translation( 'Kafka Historical Settings' ),
            'fields'                => [
                [
                    'key'               => 'field_kafka_start_date',
                    'label'             => self::get_translation( 'Start Date' ),
                    'name'              => 'start_date',
                    'type'              => 'date_picker',
                    'display_format'    => 'Y-m-d',
                    'return_format'     => 'Y-m-d',
                    'first_day'         => 1,
                ],
                [
                    'key'               => 'field_kafka_end_date',
                    'label'             => self::get_translation( 'End Date' ),
                    'name'              => 'end_date',
                    'type'              => 'date_picker',
                    'display_format'    => 'Y-m-d',
                    'return_format'     => 'Y-m-d',
                    'first_day'         => 1,
                ],
            ],
            'location'              => [
                [
                    [
                        'param'     => 'options_page',
                        'operator'  => '==',
                        'value'     => 'acf-options-kafka-historical',
                    ],
                ],
            ],
        ]);
    }

    public static function save_options( $post_id ): void {
        if ( $post_id !== 'options' ) {
            return;
        }

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            self::store_notice(
                __( 'You do not have permission to do this!', 'fitsmartio-en' ),
                'error'
            );

            return;
        }

        $start = get_field( 'start_date', 'option' );
        $end   = get_field( 'end_date',   'option' );

        if ( empty( $start ) || empty( $end ) ) {
            self::store_notice(
                __( 'Both start and end dates are required.', 'fitsmartio-en' ),
                'error'
            );

            return;
        }

        try {
            $count = Class_Woocommerce_Kafka_Historical::sendHistoricalData( $start, $end );

            $message = sprintf(
                _n(
                    'Kafka: %d order has been sent.',
                    'Kafka: %d orders have been sent.',
                    $count,
                    'fitsmartio-en'
                ),
                $count
            );

            self::store_notice( $message, 'success' );
        } catch ( \Exception $e ) {
            self::store_notice(
                sprintf(
                    __( 'Kafka export failed: %s', 'fitsmartio-en' ),
                    $e->getMessage()
                ),
                'error'
            );
        }
    }

    public static function admin_notice(): void {
        $notice = get_option( self::NOTICE_OPTION );

        if ( empty( $notice['message'] ) ) {
            return;
        }

        $screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
        if ( $screen && $screen->id !== 'tools_page_acf-options-kafka-historical' ) {
            return;
        }

        if ( function_exists( 'wp_admin_notice' ) ) {
            wp_admin_notice(
                $notice['message'],
                [
                    'type'        => $notice['type']      ?? 'info',
                    'dismissible' => $notice['dismiss']   ?? true,
                ]
            );
        } else {
            printf(
                '<div class="notice notice-%1$s%3$s"><p>%2$s</p></div>',
                esc_attr( $notice['type'] ?? 'info' ),
                esc_html( $notice['message'] ),
                empty( $notice['dismiss'] ) ? '' : ' is-dismissible'
            );
        }

        delete_option( self::NOTICE_OPTION );
    }

    private static function store_notice(
        string  $message,
        string  $type        = 'info',
        bool    $dismissible = true
    ): void {

        update_option(
            self::NOTICE_OPTION,
            [
                'message'  => $message,
                'type'     => $type,
                'dismiss'  => $dismissible,
            ],
            false
        );
    }

    private static function get_translation(string $text): string
    {
        if (function_exists('pll__')) {
            return pll__($text);
        }

        return __($text);
    }
}
