<?php

declare(strict_types=1);

namespace Affilyads\WoocommerceKafkaIntegration\Includes\Connectors;

use GuzzleHttp\Client;
use RdKafka\Conf;
use RdKafka\Producer;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class ClassWoocommerceKafkaRawConnector
{

	private bool $isEnabled = false;

	protected static $instance;
	private ?Client $schemaRegistryClient = null;
	private bool $checkedDependencies = false;
	private string $brokers;
	private string $schemaRegistryUrl;
	private float $timeout = 5.0;
	private int $logLevel;
	private ?Producer $producer = null;


	protected const REQUIRED_CONFIG = [
		'WP_KAFKA_RAW_BROKERS',
		'WP_KAFKA_RAW_LOG_LEVEL',
		'WP_KAFKA_RAW_CHANNEL_CODE',
		'WP_KAFKA_SCHEMA_REGISTRY_URL'
	];

    /**
     * @throws \Exception
     */
    public function __construct()
	{
		if (defined('WP_KAFKA_ENABLED_RAW')) {
			$this->isEnabled = (bool) WP_KAFKA_ENABLED_RAW;
		}

		if (defined('WP_KAFKA_RAW_TIMEOUT')) {
			$this->timeout = (float) WP_KAFKA_RAW_TIMEOUT;
		}

		if ($this->isEnabled) {
			$this->checkDependencies();

			if ($this->checkedDependencies) {
				$this->brokers = WP_KAFKA_RAW_BROKERS;
				$this->schemaRegistryUrl = WP_KAFKA_SCHEMA_REGISTRY_URL;
				$this->logLevel = WP_KAFKA_RAW_LOG_LEVEL;
			}
		}
	}

	public static function getInstance(): self
	{
		if (is_null(self::$instance)) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	private function initConnection(): void
	{
		$this->config = new Conf();
		$this->config->set('log_level', (string)$this->logLevel);

		if ( $this->logLevel === LOG_DEBUG) {
			$this->config->set('debug', 'all');
		}

		$this->config->set('enable.idempotence', 'true');
		$this->config->set('acks', 'all');
		$this->config->set('max.in.flight.requests.per.connection', '5');

		$this->producer = new Producer($this->config);
		$this->producer->addBrokers($this->brokers);
	}

	public function getProducer(): Producer
	{
		if (null == $this->producer) {
			$this->initConnection();
		}

		return $this->producer;
	}

	public function getSchemaRegistryClient(): Client {
		if (!$this->schemaRegistryClient) {
			$this->schemaRegistryClient = new Client([
				'base_uri' => $this->schemaRegistryUrl,
				'timeout'  => $this->timeout,
			]);
		}

		return $this->schemaRegistryClient;
	}

    /**
     * @throws \Exception
     */
    private function checkDependencies(): void
	{
		if (!$this->isEnabled) {
			return;
		}

		$this->checkedDependencies = true;

		foreach (self::REQUIRED_CONFIG as $config) {
			if (!defined($config) || null === constant($config)) {
				$this->checkedDependencies = false;
				throw new \Exception("$config is not defined");
			}
		}
	}

	public function isCheckedDependencies(): bool
	{
		return $this->checkedDependencies;
	}

}