<?php

declare(strict_types=1);

namespace woocommerce_kafka_integration\includes\connectors;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

final class Class_Woocommerce_Kafka_Connector
{

    protected static $instance;
	private bool $checkedDependencies = false;
    protected $producer;
    protected $config;

    protected $brokers;
    protected $log_level;

    protected const REQUIRED_CONFIG = [
        'WP_KAFKA_BROKERS',
        'WP_KAFKA_LOG_LEVEL'
    ];

    public function __construct()
    {
        try {
            $this->checkDependencies();
			$this->checkedDependencies = true;
        } catch (\Exception $e) {
            global $bugsnagWordpress;

            if ($bugsnagWordpress) {
                $bugsnagWordpress->notifyException($e);
            }

            error_log('WKI | checkDependencies | ' . $e->getMessage());
            error_log('WKI | checkDependencies | ' . $e->getTraceAsString());
        }

        $this->brokers = WP_KAFKA_BROKERS;
        $this->log_level = WP_KAFKA_LOG_LEVEL;
    }

    public static function getInstance(): self
    {
        if (is_null(self::$instance)) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    /**
     * @throws \Exception
     */
    private function checkDependencies(): void
    {

        if (!class_exists('\\RdKafka\\Conf')) {
            throw new \Exception('RdKafka not installed');
        }

        if (!class_exists('\\RdKafka\\Producer')) {
            throw new \Exception('RdKafka not installed');
        }

        foreach (self::REQUIRED_CONFIG as $config) {
            if (!defined($config)) {
                throw new \Exception("$config is not defined");
            }
        }

    }

    private function initConnection(): void
    {
        $this->config = new \RdKafka\Conf();
        $this->config->set('log_level', (string)$this->log_level);

        if ($this->log_level === LOG_DEBUG) {
            $this->config->set('debug', 'all');
        }

        $this->producer = new \RdKafka\Producer($this->config);
        $this->producer->addBrokers($this->brokers);
    }

    public function getProducer(): \RdKafka\Producer
    {
        if (null == $this->producer) {
            $this->initConnection();
        }

        return $this->producer;
    }

	public function isCheckedDependencies(): bool {
		return $this->checkedDependencies;
	}

}