<?php

declare(strict_types=1);

namespace Affilyads\WoocommerceKafkaIntegration\Includes\Connectors;

use Exception;
use RdKafka\Conf;
use RdKafka\Producer;
use function Symfony\Component\String\b;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class ClassWoocommerceKafkaConnector
{

	private bool $enabledMain = false;

	protected static $instance;
	private bool $checkedDependencies = false;
	protected $producer;
	protected $config;

	protected $brokers;
	protected $log_level;
	protected const REQUIRED_CONFIG = [
		'WP_KAFKA_BROKERS',
		'WP_KAFKA_LOG_LEVEL'
	];

	public function __construct()
	{
		if (defined('WP_KAFKA_ENABLED_MAIN')) {
			$this->enabledMain = (bool) WP_KAFKA_ENABLED_MAIN;
		}

		if ($this->enabledMain) {
			$this->checkDependencies();

			$this->brokers = WP_KAFKA_BROKERS;
			$this->log_level = WP_KAFKA_LOG_LEVEL;
		}
	}

	public static function getInstance(): self
	{
		if (is_null(self::$instance)) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	/**
	 * @throws Exception
	 */
	private function checkDependencies(): void
	{

		if (!class_exists('\\RdKafka\\Conf')) {
			throw new Exception('RdKafka not installed');
		}

		if (!class_exists('\\RdKafka\\Producer')) {
			throw new Exception('RdKafka not installed');
		}

		$this->checkedDependencies = true;

		foreach (self::REQUIRED_CONFIG as $config) {
			if (!defined($config) || null === constant($config)) {
				error_log("$config is not defined");
				$this->checkedDependencies = false;
				break;
			}
		}

	}

	private function initConnection(): void
	{
		$this->config = new Conf();
		$this->config->set('log_level', (string)$this->log_level);

		if ($this->log_level === LOG_DEBUG) {
			$this->config->set('debug', 'all');
		}

		$this->config->set('enable.idempotence', 'true');
		$this->config->set('acks', 'all');
		$this->config->set('max.in.flight.requests.per.connection', '5');

		$this->producer = new Producer($this->config);
		$this->producer->addBrokers($this->brokers);
	}

	public function getProducer(): Producer
	{
		if (null == $this->producer) {
			$this->initConnection();
		}

		return $this->producer;
	}

	public function isCheckedDependencies(): bool {
		return $this->checkedDependencies;
	}
}