<?php

declare(strict_types=1);

namespace woocommerce_kafka_integration\includes;

final class Class_Woocommerce_Kafka_Core
{

	protected static ?Class_Woocommerce_Kafka_Core $instance = null;

	protected connectors\Class_Woocommerce_Kafka_Connector $connector;

	public static function getInstance(): self
	{
		if (null === self::$instance) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	public function __construct()
	{
		$this->connector = \woocommerce_kafka_integration\includes\connectors\Class_Woocommerce_Kafka_Connector::getInstance();

		if ($this->connector->isCheckedDependencies()) {
			$this->addEventListeners();
			$this->registerScheduleActions();
			$this->registerAjax();
		}
	}

	/**
	 * @param int $order_id
	 * @param mixed $order Unused property for correctly action work
	 * @return void
	 */
	public function handleWoocommerceOrderUpdated(int $order_id, mixed $order): void
	{
		$cache_key = "order_update_$order_id";

		wp_cache_delete($cache_key);

		as_unschedule_action("send_kafka_order_update", [$order_id]);

		as_schedule_single_action(time() + 5, "send_kafka_order_update", [$order_id]);
	}

	/**
	 * @param int $order_id
	 * @param mixed $order Unused property for correctly action work
	 * @return void
	 */
	public function handleWoocommerceOrderCreated(int $order_id, mixed $order): void
	{
		$cache_key = "order_create_$order_id";

		wp_cache_delete($cache_key);

		as_unschedule_action("send_kafka_order_create", [$order_id]);

		as_schedule_single_action(time() + 5, "send_kafka_order_create", [$order_id]);
	}

	private function registerScheduleActions(): void
	{
		add_action( 'send_kafka_order_create', [
			new \woocommerce_kafka_integration\includes\topics\Class_Woocommerce_Kafka_Order_Created_Topic($this->connector),
			'send'
		], 10, 1 );

		add_action( 'send_kafka_order_update', [
			new \woocommerce_kafka_integration\includes\topics\Class_Woocommerce_Kafka_Order_Updated_Topic($this->connector),
			'send'
		], 10, 1 );
	}

	private function addEventListeners(): void
	{
		add_action( 'woocommerce_new_order', [$this, 'handleWoocommerceOrderCreated'], 10, 2 );

		add_action('woocommerce_update_order', [$this, 'handleWoocommerceOrderUpdated'], 10, 2);
		add_action('woocommerce_order_refunded', [$this, 'handleWoocommerceOrderUpdated'], 10, 2);
	}

	private function registerAjax(): void
	{
		add_action('wp_ajax_nopriv_kafkasendhistoricalorder', [new Class_Woocommerce_Kafka_Historical(), 'sendHistoricalData'], PHP_INT_MAX, 2);
		add_action('wp_ajax_kafkasendhistoricalorder', [new Class_Woocommerce_Kafka_Historical(), 'sendHistoricalData'], PHP_INT_MAX, 2);
	}

}