<?php

declare( strict_types=1 );

namespace Affilyads\WoocommerceKafkaIntegration\Includes\Topics;

use Affilyads\WoocommerceKafkaIntegration\Includes\Connectors\ClassWoocommerceKafkaConnector;
use Affilyads\WoocommerceKafkaIntegration\Includes\Connectors\ClassWoocommerceKafkaRawConnector;
use GuzzleHttp\Exception\GuzzleException;

abstract class ClassWoocommerceKafkaAbstractTopic {

	private ClassWoocommerceKafkaConnector $connector;
	private ?ClassWoocommerceKafkaRawConnector $rawConnector;
	private array $topics = [];

	abstract protected function getTopicName(): string;

	abstract protected function prepareData(): array;

	abstract public function send(): void;

	protected function getRawTopicName(): string {
		if ( defined( 'WP_KAFKA_RAW_ORDERS_TOPIC_NAME' ) && null !== constant( 'WP_KAFKA_RAW_ORDERS_TOPIC_NAME' ) ) {
			return WP_KAFKA_RAW_ORDERS_TOPIC_NAME;
		}

		return 'orders_raw';
	}

	protected function getDlqTopicName(): string {
		if ( defined( 'WP_KAFKA_RAW_ORDERS_DLQ_TOPIC_NAME' ) && null !== constant( 'WP_KAFKA_RAW_ORDERS_DLQ_TOPIC_NAME' ) ) {
			return WP_KAFKA_RAW_ORDERS_DLQ_TOPIC_NAME;
		}

		return 'orders_dlq';
	}

	public function __construct( ClassWoocommerceKafkaConnector $connector, ?ClassWoocommerceKafkaRawConnector $rawConnector = null ) {
		$this->connector    = $connector;
		$this->rawConnector = $rawConnector;

		$this->initTopics();
	}

	private function initTopics(): void {
		if ( ! $this->connector->isCheckedDependencies() ) {
			return;
		}

		$topicNames = array_filter( array_map( 'trim', explode( ',', $this->getTopicName() ) ) );

		foreach ( $topicNames as $topicName ) {
			$this->topics[] = $this->connector->getProducer()->newTopic( $topicName );
		}
	}

	protected function sendData( array $preparedData ): void {
		if ( ! $this->connector->isCheckedDependencies() ) {
			return;
		}

		foreach ( $this->topics as $topic ) {
			$topic->produce( RD_KAFKA_PARTITION_UA, 0, json_encode( $preparedData ), (string) $preparedData['id'] ?? null );
		}

		$this->connector->getProducer()->flush( 10000 );
	}

	protected function sendRawData( array $preparedData ): void {
		if ( ! $this->rawConnector || ! $this->rawConnector->isCheckedDependencies() ) {
			return;
		}

		$clusterId = WP_KAFKA_RAW_CLUSTER_ID;

		$preparedData['channel_code'] = WP_KAFKA_RAW_CHANNEL_CODE;

		$preparedData['date_created']      .= 'Z';
		$preparedData['date_created_gmt']  .= 'Z';

		$preparedData['date_modified']     .= 'Z';
		$preparedData['date_modified_gmt'] .= 'Z';

		$preparedData['date_completed']    .= 'Z';
		$preparedData['date_completed_gmt'].= 'Z';

		$preparedData['date_paid']         .= 'Z';
		$preparedData['date_paid_gmt']     .= 'Z';

		$rawTopicName = $this->getRawTopicName();

		$body = [
			'key'   => [
				'data' => [
					'id'   => $preparedData['id'] ?? null,
					'code' => $preparedData['billing']['country']
				]
			],
			'value' => [
				'data' => $preparedData
			]
		];


		try {
			$response = $this->rawConnector->getClient()
			                               ->post( "/v3/clusters/$clusterId/topics/$rawTopicName/records", [
				                               'body'    => json_encode( $body ),
				                               'headers' => [
					                               'Content-Type' => 'application/json',
				                               ]
			                               ] );

			$responseJson = json_decode( $response->getBody()->getContents(), true );

			error_log(json_encode($responseJson));

			if ( isset($responseJson['error_code']) && $responseJson['error_code'] !== 200 ) {
				$this->sendDataToDlq( $body );
			}
		} catch ( GuzzleException $e ) {
			$this->sendDataToDlq( $body );
		}
	}

	protected function sendDataToDlq(array $body): void {
		error_log('sendDataToDlq called');

		$clusterId = WP_KAFKA_RAW_CLUSTER_ID;
		$dlqTopicName = $this->getDlqTopicName();

		$body['headers'] = [
			[
				'name' => 'dlq_reason',
				'value' => base64_encode( 'schema_validation' )
			],
			[
				'name' => 'dlq_timestamp',
				'value' => base64_encode( (string) time() )
			],
			[
				'name' => 'dlq_attempt_counter',
				'value' => base64_encode( "1" )
			]
		];

		$body['key']['type'] = 'JSON';
		$body['value']['type'] = 'JSON';

		try {
			$response = $this->rawConnector->getClient()->post( "/v3/clusters/$clusterId/topics/$dlqTopicName/records", [
				'body'    => json_encode( $body ),
				'headers' => [
					'Content-Type' => 'application/json'
				]
			] );

			error_log( 'sendDataToDlq' . $response->getBody()->getContents() );
		} catch ( GuzzleException $e ) {
			error_log( 'sendDataToDlq catch' . $e->getMessage() );
		}
	}

}