<?php

declare(strict_types=1);

namespace Affilyads\WoocommerceKafkaIntegration\Includes\Connectors;

use GuzzleHttp\Client;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

final class ClassWoocommerceKafkaRawConnector
{

	private bool $isEnabled = false;

	protected static $instance;
	private ?Client $client = null;
	private bool $checkedDependencies = false;
	private string $url;
	private float $timeout = 5.0;


	protected const REQUIRED_CONFIG = [
		'WP_KAFKA_RAW_URL',
		'WP_KAFKA_RAW_LOG_LEVEL',
		'WP_KAFKA_RAW_CHANNEL_CODE',
		'WP_KAFKA_RAW_CLUSTER_ID'
	];

	public function __construct()
	{
		if (defined('WP_KAFKA_ENABLED_RAW')) {
			$this->isEnabled = (bool) WP_KAFKA_ENABLED_RAW;
		}

		if (defined('WP_KAFKA_RAW_TIMEOUT')) {
			$this->timeout = (float) WP_KAFKA_RAW_TIMEOUT;
		}

		if ($this->isEnabled) {
			$this->checkDependencies();

			if ($this->checkedDependencies) {
				$this->url = WP_KAFKA_RAW_URL;
			}
		}
	}

	public static function getInstance(): self
	{
		if (is_null(self::$instance)) {
			self::$instance = new self;
		}

		return self::$instance;
	}

	public function getClient(): Client {
		if (!$this->client) {
			$this->client = new Client([
				'base_uri' => $this->url,
				'timeout'  => $this->timeout,
			]);
		}

		return $this->client;
	}

	private function checkDependencies(): void
	{
		if (!$this->isEnabled) {
			return;
		}

		$this->checkedDependencies = true;

		foreach (self::REQUIRED_CONFIG as $config) {
			if (!defined($config) || null === constant($config)) {
				error_log("$config is not defined");
				$this->checkedDependencies = false;
				break;
			}
		}
	}

	public function isCheckedDependencies(): bool
	{
		return $this->checkedDependencies;
	}

}