<?php

use Fitsmart\PaypalParcels\Config;
use Fitsmart\PaypalParcels\ExternalService\TrackingNoteParser;
use PHPUnit\Framework\TestCase;

class TrackingNoteParserTest extends TestCase
{
    private array $testCases = [
        // Huboo cases
        [
            'note' => 'Huboo dispatched order - Tracking: RS564825199DE Url: https://packet.deutschepost.com/web/portal-europe/packet_traceit?barcode=RS564825199DE',
            'number' => 'RS564825199DE',
            'paypal_carrier' => 'DEUTSCHE_DE',
            'parcelpanel_carrier' => 'deutsche-post',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: H01HYA0037675029 Url: https://www.evri.com/track/parcel/H01HYA0037675029/details',
            'number' => 'H01HYA0037675029',
            'paypal_carrier' => 'HERMESWORLD_UK',
            'parcelpanel_carrier' => 'hermes-uk',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: QD574652997GB Url: https://www.royalmail.com/track-your-item#/tracking-results/QD574652997GB',
            'number' => 'QD574652997GB',
            'paypal_carrier' => 'ROYAL_MAIL',
            'parcelpanel_carrier' => 'royal-mail',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: LA192766734NL Url: https://postnl.nl/tracktrace/?D=IE&B=LA192766734NL&P=K67AP90&T=C',
            'number' => 'LA192766734NL',
            'paypal_carrier' => 'POSTNL_INTERNATIONAL',
            'parcelpanel_carrier' => 'postnl-parcels',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: HUB1080GB97156861101 Url: https://fictracking.fedex.com/?MyFICNumber=HUB1080GB97156861101',
            'number' => 'HUB1080GB97156861101',
            'paypal_carrier' => 'FEDEX_FR',
            'parcelpanel_carrier' => 'fedex',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: 285751947317 Url: https://www.fedex.com/apps/fedextrack/index.html?tracknumbers=285751947317',
            'number' => '285751947317',
            'paypal_carrier' => 'FEDEX_FR',
            'parcelpanel_carrier' => 'fedex',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: 01045231696737 Url: https://tracking.dpd.de/status/de_DE/parcel/01045231696737',
            'number' => '01045231696737',
            'paypal_carrier' => 'DPD',
            'parcelpanel_carrier' => 'dpd-de',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: YY28IYK6 Url: https://gls-group.eu/track/YY28IYK6',
            'number' => 'YY28IYK6',
            'paypal_carrier' => 'GLS',
            'parcelpanel_carrier' => 'gls',
        ],
        // Universal case
        [
            'note' => 'Huboo dispatched order - Tracking: HG87T4G Url: https://other-parcel.com/track/en/HG87t4g',
            'number' => 'HG87T4G',
            'paypal_carrier' => 'OTHER',
            'parcelpanel_carrier' => '',
        ],
        // ShipBob cases
        [
            'note' => 'Order has been shipped by Spring Tracked (https://mailingtechnology.com/tracking/?tn=LR514453469NL)',
            'number' => 'LR514453469NL',
            'paypal_carrier' => 'OTHER',
            'paypal_carrier_name_other' => 'Spring Tracked',
            'parcelpanel_carrier' => 'mailing-technology',
        ],
        [
            'note' => 'Order has been shipped by UPS (http://wwwapps.ups.com/WebTracking/track?track=yes&trackNums=1ZRJ70536825413991&loc=en_us)',
            'number' => '1ZRJ70536825413991',
            'paypal_carrier' => 'UPS',
            'parcelpanel_carrier' => 'ups',
        ],
        [
            'note' => 'Order has been shipped by Deutsche Post International Tracked (https://www.packet.deutschepost.com/webapp/public/packet_traceit.xhtml?barcode=LY833056401DE)',
            'number' => 'LY833056401DE',
            'paypal_carrier' => 'DEUTSCHE_DE',
            'parcelpanel_carrier' => 'deutsche-post',
        ],

    ];

    public function testVariousTrackingNotes()
    {
        $parser = new TrackingNoteParser();
        foreach ($this->testCases as $case) {
            $data = $parser->parse($case['note'], Config::getConfigPart('parse_data'));

            dump($data);

            $this->assertEquals($case['number'], $data['tracking_number']);
            $this->assertEquals($case['paypal_carrier'], $data['paypal_carrier']);
            $this->assertEquals($case['parcelpanel_carrier'], $data['parcelpanel_carrier']);

            $configPaypalPart = Config::getConfigPart('paypal') ?? [];
            $otherCarrier = $configPaypalPart['other_carrier'] ?? '';

            if ($data['paypal_carrier'] === $otherCarrier) {
                $this->assertArrayHasKey('paypal_carrier_name_other', $data);
            }
        }
    }

}
