<?php

use Fitsmart\PaypalParcels\Config;
use Fitsmart\PaypalParcels\ExternalService\TrackingNoteParser;
use PHPUnit\Framework\TestCase;

class TrackingNoteParserTest extends TestCase
{
    private array $testCases = [
        // Huboo cases
        [
            'note' => 'Huboo dispatched order - Tracking: RS564825199DE Url: https://packet.deutschepost.com/web/portal-europe/packet_traceit?barcode=RS564825199DE',
            'number' => 'RS564825199DE',
            'carrier' => 'DEUTSCHE_DE',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: H01HYA0037675029 Url: https://www.evri.com/track/parcel/H01HYA0037675029/details',
            'number' => 'H01HYA0037675029',
            'carrier' => 'HERMESWORLD_UK',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: QD574652997GB Url: https://www.royalmail.com/track-your-item#/tracking-results/QD574652997GB',
            'number' => 'QD574652997GB',
            'carrier' => 'ROYAL_MAIL',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: LA192766734NL Url: https://postnl.nl/tracktrace/?D=IE&B=LA192766734NL&P=K67AP90&T=C',
            'number' => 'LA192766734NL',
            'carrier' => 'POSTNL_INTERNATIONAL',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: HUB1080GB97156861101 Url: https://fictracking.fedex.com/?MyFICNumber=HUB1080GB97156861101',
            'number' => 'HUB1080GB97156861101',
            'carrier' => 'FEDEX_FR',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: 285751947317 Url: https://www.fedex.com/apps/fedextrack/index.html?tracknumbers=285751947317',
            'number' => '285751947317',
            'carrier' => 'FEDEX_FR',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: 01045231696737 Url: https://tracking.dpd.de/status/de_DE/parcel/01045231696737',
            'number' => '01045231696737',
            'carrier' => 'DPD',
        ],
        [
            'note' => 'Huboo dispatched order - Tracking: YY28IYK6 Url: https://gls-group.eu/track/YY28IYK6',
            'number' => 'YY28IYK6',
            'carrier' => 'GLS',
        ],
        // Universal case
        [
            'note' => 'Huboo dispatched order - Tracking: HG87T4G Url: https://other-parcel.com/track/en/HG87t4g',
            'number' => 'HG87T4G',
            'carrier' => 'OTHER',
        ],
        // ShipBob cases
        [
            'note' => 'Order has been shipped by Spring Tracked (https://mailingtechnology.com/tracking/?tn=LR514453469NL)',
            'number' => 'LR514453469NL',
            'carrier' => 'OTHER',
        ],
        [
            'note' => 'Order has been shipped by UPS (http://wwwapps.ups.com/WebTracking/track?track=yes&trackNums=1ZRJ70536825413991&loc=en_us)',
            'number' => '1ZRJ70536825413991',
            'carrier' => 'UPS',
        ],
        [
            'note' => 'Order has been shipped by Deutsche Post International Tracked (https://www.packet.deutschepost.com/webapp/public/packet_traceit.xhtml?barcode=LY833056401DE)',
            'number' => 'LY833056401DE',
            'carrier' => 'DEUTSCHE_DE',
        ],

    ];

    public function testVariousTrackingNotes()
    {
        $parser = new TrackingNoteParser();
        foreach ($this->testCases as $case) {
            $data = $parser->parse($case['note'], Config::getConfigPart('parse_data'));

            $this->assertEquals($case['number'], $data['tracking_number']);
            $this->assertEquals($case['carrier'], $data['carrier']);

            $configPaypalPart = Config::getConfigPart('paypal') ?? [];
            $otherCarrier = $configPaypalPart['other_carrier'] ?? '';

            if($data['carrier'] === $otherCarrier) {
                $this->assertArrayHasKey('carrier_name_other', $data);
            }
        }
    }

}