<?php

namespace PaymentWebhooks\Application\Admin;

use PaymentWebhooks\Contract\WordPress\HookableInterface;

final class SettingsPage implements HookableInterface
{
    public const MENU_SLUG = 'payment-webhooks';

    private string $pluginBasename;
    public function __construct(string $pluginBasename)
    {
        $this->pluginBasename = $pluginBasename;
    }


    public function hooks(): void
    {
        add_action('acf/init', [$this, 'registerPageAndFields']);
        add_filter(
            "plugin_action_links_{$this->pluginBasename}",
            [$this, 'addSettingsLink']
        );
    }

    public function registerPageAndFields(): void
    {
        if (!function_exists('acf_add_options_page')) {
            return;
        }

        acf_add_options_page([
            'page_title' => __('Payment Webhooks Settings', 'payment-webhooks'),
            'menu_title' => __('Payment Webhooks', 'payment-webhooks'),
            'menu_slug' => self::MENU_SLUG,
            'capability' => 'manage_options',
            'redirect' => false,
        ]);

        acf_add_local_field_group([
            'key' => 'group_pw_settings',
            'title' => __('Payment Webhooks Config', 'payment-webhooks'),
            'fields' => $this->getFieldDefinitions(),
            'location' => [
                [
                    [
                        'param' => 'options_page',
                        'operator' => '==',
                        'value' => self::MENU_SLUG,
                    ]
                ],
            ],
        ]);
    }

    public function addSettingsLink(array $links): array
    {
        $url = admin_url('admin.php?page=' . self::MENU_SLUG);
        $links[] = '<a href="' . esc_url($url) . '">' . esc_html__('Settings', 'default') . '</a>';
        return $links;
    }

    private function getFieldDefinitions(): array
    {
        return [
            // --- Global --- //
            [
                'key'   => 'field_pw_async_update',
                'label' => 'PW_ASYNC_UPDATE',
                'name'  => 'PW_ASYNC_UPDATE',
                'type'  => 'true_false',
                'ui'    => 1,
            ],

            // --- Adyen --- //
            [
                'key'      => 'field_pw_adyen_user',
                'label'    => 'PW_ADYEN_USER',
                'name'     => 'PW_ADYEN_USER',
                'type'     => 'text',
                'required' => 1,
            ],
            [
                'key'      => 'field_pw_adyen_pass',
                'label'    => 'PW_ADYEN_PASS',
                'name'     => 'PW_ADYEN_PASS',
                'type'     => 'password',
                'required' => 1,
            ],
            [
                'key'      => 'field_pw_adyen_hmac',
                'label'    => 'PW_ADYEN_HMAC_KEY',
                'name'     => 'PW_ADYEN_HMAC_KEY',
                'type'     => 'password',
                'required' => 1,
            ],
            [
                'key'   => 'field_pw_adyen_log',
                'label' => 'PW_ADYEN_LOG',
                'name'  => 'PW_ADYEN_LOG',
                'type'  => 'true_false',
                'ui'    => 1,
            ],

            // --- PayPal --- //
            [
                'key'   => 'field_pw_paypal_log',
                'label' => 'PW_PAYPAL_LOG',
                'name'  => 'PW_PAYPAL_LOG',
                'type'  => 'true_false',
                'ui'    => 1,
            ],
            [
                'key'      => 'field_pw_paypal_webhook',
                'label'    => 'PW_PAYPAL_WEBHOOK_ID',
                'name'     => 'PW_PAYPAL_WEBHOOK_ID',
                'type'     => 'text',
                'required' => 1,
            ],
            [
                'key'      => 'field_pw_paypal_cert',
                'label'    => 'PW_PAYPAL_CERT_FOLDER_NAME',
                'name'     => 'PW_PAYPAL_CERT_FOLDER_NAME',
                'type'     => 'text',
                'required' => 1,
            ],
        ];
    }

}
