<?php

namespace FKCart\Pro;

use FKCart\Includes\Data;
use FKCart\Includes\Front as Front;
if ( ! class_exists( '\FKCart\Pro\Upsells' ) ) {
	#[\AllowDynamicProperties]
	class Upsells {
		public static $stats_table = 'fk_cart_stats';
		private static $instance = null;

		private function __construct() {
			add_action( 'fkcart_settings_saved', [ $this, 'maybe_create_table' ] );
			$data = Data::get_db_settings();
			if ( ! isset( $data['enable_cart'] ) || 0 === intval( $data['enable_cart'] ) ) {
				return false;
			}
			add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'woocommerce_create_order_line_item' ], 999999, 3 );
			add_action( 'bwf_normalize_contact_meta_after_save', [ $this, 'insert_stats' ], 10, 3 );
			add_action( 'woocommerce_order_fully_refunded', array( $this, 'fully_refunded_process' ) );
			add_action( 'woocommerce_order_partially_refunded', array( $this, 'partially_refunded_process' ), 10, 2 );
		}

		/**
		 * @return Upsells
		 */
		public static function getInstance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;

		}

		/**
		 * Maybe create upsell stats table on settings save
		 *
		 * @param $data
		 *
		 * @return void
		 */
		public function maybe_create_table( $data ) {
			if ( 1 !== intval( $data['enable_upsells'] ) || $this->is_stats_table_created() ) {
				return;
			}

			$this->create_table();
		}

		/**
		 * Check if stats table created
		 *
		 * @return bool
		 */
		public function is_stats_table_created() {
			$db_options = get_option( 'fkcart_db_options', [] );
			if ( isset( $db_options['stats_table'] ) && 1 === intval( $db_options['stats_table'] ) ) {
				return true;
			}

			return false;
		}

		/**
		 * Create stats table
		 *
		 * @return void
		 */
		public function create_table() {
			require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
			global $wpdb;

			$collate = '';
			if ( $wpdb->has_cap( 'collation' ) ) {
				$collate = $wpdb->get_charset_collate();
			}

			$table_name = $wpdb->prefix . self::$stats_table;

			$creationSQL = "CREATE TABLE {$table_name} (
 		  `ID` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
		  `oid` bigint unsigned NOT NULL COMMENT 'Order ID',
		  `pid` bigint unsigned NOT NULL DEFAULT '0' COMMENT 'Product ID',
		  `cid` bigint unsigned NOT NULL COMMENT 'Contact ID',
		  `total` double unsigned NOT NULL DEFAULT '0',
		  `date_created` datetime NOT NULL,
		  `date_created_gmt` datetime NOT NULL,
          `status` tinyint(1) unsigned NOT NULL,
 		  PRIMARY KEY (ID),
		  KEY ID (ID),
		  KEY oid (oid),
		  KEY pid (pid),
		  KEY cid (cid),
		  KEY status (status)
		) $collate;";
			\dbDelta( $creationSQL );

			$db_options['stats_table'] = 1;
			update_option( 'fkcart_db_options', $db_options, false );
		}

		/**
		 * @param $item \WC_Order_Item
		 * @param $cart_item_key
		 * @param $values
		 */
		public function woocommerce_create_order_line_item( $item, $cart_item_key, $values ) {
			if ( ! isset( $values['_fkcart_upsell'] ) ) {
				return;
			}

			$item->add_meta_data( '_fkcart_upsell', 'yes' );
		}

		/**
		 * Insert Upsell Sell into reporting table
		 *
		 * @param $bwf_contact \WooFunnels_Contact
		 * @param $order_id int Order ID
		 * @param $order \WC_Order
		 *
		 * @return void
		 */
		public function insert_stats( $bwf_contact, $order_id, $order ) {
			$is_upsell_enabled = Data::is_upsells_enabled();
			if ( false === $is_upsell_enabled ) {
				return;
			}

			if ( ! $order instanceof \WC_Order ) {
				return;
			}

			global $wpdb;

			$items = $order->get_items();
			foreach ( $items as $item ) {
				/**
				 * @var $item \WC_Order_Item
				 */
				$is_upsell = $item->get_meta( '_fkcart_upsell' );
				if ( 'yes' !== strval( $is_upsell ) ) {
					continue;
				}

				$total = \BWF_Plugin_Compatibilities::get_fixed_currency_price_reverse( $item['total'], \BWF_WC_Compatibility::get_order_currency( $order ) );
				$wpdb->insert( $wpdb->prefix . self::$stats_table, [
					'oid'              => $order_id,
					'cid'              => $bwf_contact->get_id(),
					'pid'              => $item['product_id'],
					'status'           => 1,
					'total'            => $total,
					'date_created'     => current_time( 'mysql' ),
					'date_created_gmt' => current_time( 'mysql', true )
				], [ '%d', '%d', '%d', '%d', '%f', '%s', '%s' ] );
			}
		}

		/**
		 * Mark refunded product items status refunded
		 *
		 * @param $order_id
		 * @param $refund_id
		 *
		 * @return void
		 */
		public function partially_refunded_process( $order_id, $refund_id ) {
			$order = wc_get_order( $order_id );
			if ( ! $order instanceof \WC_Order ) {
				return;
			}

			/** Check if order status refunded */
			if ( $order->has_status( 'refunded' ) ) {
				$this->fully_refunded_process( $order_id );

				return;
			}

			$refund = wc_get_order( $refund_id );
			if ( ! $refund instanceof \WC_Order_Refund ) {
				return;
			}

			$items = $refund->get_items();
			if ( 0 === count( $items ) ) {
				return;
			}

			global $wpdb;
			/** Process each line item */
			foreach ( $items as $refund_item ) {
				$item_id = $refund_item->get_meta( '_refunded_item_id', true );
				if ( empty( $item_id ) ) {
					/** No cart upsell product */
					continue;
				}

				$item = $order->get_item( $item_id );
				if ( ! $item instanceof \WC_Order_Item ) {
					continue;
				}

				$is_upsell = $item->get_meta( '_fkcart_upsell' );
				if ( 'yes' !== strval( $is_upsell ) ) {
					continue;
				}

				$wpdb->update( $wpdb->prefix . self::$stats_table, [ 'status' => 0 ], [
					'oid' => $order_id,
					'pid' => $refund_item['product_id']
				] );
			}
		}

		/**
		 * Mark refunded order all product items status as refunded
		 *
		 * @param $order_id
		 *
		 * @return void
		 */
		public function fully_refunded_process( $order_id ) {
			global $wpdb;

			$wpdb->update( $wpdb->prefix . self::$stats_table, [ 'status' => 0 ], [ 'oid' => $order_id ] );
		}

		/**
		 * Get upsell products for cart
		 *
		 * @return array
		 */
		public function get_upsell_products() {
			/** Validate */
			if ( Plugin::valid_l() === false ) {
				return [];
			}

			$upsell_ids = $this->get_upsell_ids();
			if ( empty( $upsell_ids ) ) {
				return $upsell_ids;
			}

			$max_upsell = Data::get_value( 'upsell_max_count' );
			$max_upsell = absint( $max_upsell );
			$upsells    = [];
			foreach ( $upsell_ids as $product_id ) {
				if ( empty( $product_id ) ) {
					continue;
				}
				$product = wc_get_product( $product_id );
				if ( ! $product instanceof \WC_Product || ! $product->is_in_stock() || 'publish' !== $product->get_status() ) {
					continue;
				}
				if ( $max_upsell > 0 && count( $upsells ) >= $max_upsell ) {
					break;
				}

				$upsells[ $product_id ] = Front::get_instance()->get_preview_item( $product );
			}

			return $upsells;
		}

		/**
		 * Get upsell products ids
		 *
		 * @return array
		 */
		public function get_upsell_ids() {
			$items = Front::get_instance()->get_items();
			if ( empty( $items ) ) {
				return [];
			}

			/** @var \WC_Product $_product */
			$r_type          = $this->get_recommendation_type();
			$out_puts        = [];
			$already_used    = [];
			$default_upsells = [];

			$show_default_upsells = Data::get_value( 'show_default_upsell' );
			$show_default_upsells = ( 1 === intval( $show_default_upsells ) || true === $show_default_upsells || 'true' === strval( $show_default_upsells ) );

			foreach ( $items as $item ) {
				$_product = isset( $item['product'] ) ? $item['product'] : null;
				if ( is_null( $_product ) || ! $_product instanceof \WC_Product ) {
					continue;
				}
				$product_id     = $_product->get_id();
				$already_used[] = $product_id;
				if ( fkcart_is_variation_product_type( $_product->get_type() ) ) {
					$parent_id      = $_product->get_parent_id();
					$_product       = wc_get_product( $parent_id );
					$already_used[] = $parent_id;
				}

				if ( 'both' === $r_type ) {
					$upsell_ids = array_merge( $_product->get_upsell_ids(), $_product->get_cross_sell_ids() );
				} else {
					$upsell_ids = 'upsell' === $r_type ? $_product->get_upsell_ids() : $_product->get_cross_sell_ids();
				}
				$out_puts = array_merge( $out_puts, $upsell_ids );
				unset( $_product, $upsell_ids, $is_variation, $product_id );
			}

			/** If no upsell & cross-sells found then check for default upsells OR always show default upsell setting is enabled */
			if ( empty( $out_puts ) || $show_default_upsells ) {
				$default_upsells = $this->get_default_upsells();
			}

			$out_puts = array_merge( $default_upsells, $out_puts );
			if ( empty( $out_puts ) ) {
				return [];
			}

			return array_filter( array_unique( $out_puts ), function ( $single ) use ( $already_used ) {
				return ! in_array( $single, $already_used );
			} );
		}

		/**
		 * Get upsell recommendation products types
		 *
		 * @return mixed|string
		 */
		public function get_recommendation_type() {
			return Data::get_value( 'upsell_type' );
		}

		/**
		 * Get default upsell products ids saved in the DB
		 *
		 * @return array
		 */
		public function get_default_upsells() {
			$default_upsell = Data::get_value( 'default_upsell' );
			if ( empty( $default_upsell ) ) {
				return [];
			}

			return apply_filters( 'fkcart_default_upsells', array_map( 'intval', array_column( $default_upsell, 'key' ) ) );
		}
	}
}