<?php
/**
 * Plugin Name: Everflow Thank you and S2S firing. Headless version
 * Description: v 3.6.1 - Fires Everflow via S2S
 * Version:     3.6.1
 * Author:      Den
 * License:     GPL2
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once __DIR__ . '/vendor/autoload.php';

use Dotenv\Dotenv;
use UAParser\Parser;
use App\QuasiEnums\FireStatuses;

class EverflowIntegration
{
    public const CID2_META_KEY = '_cid_2';
    public const AFFILIATE_URL_META_KEY = '_wc_order_attribution_session_entry';
    public const FIRE_STATUS_META_KEY = '_everflow_fire_status';
    private const ASYNC_POSTBACK_DELAY = 10;
    private static ?self $instance = null;
    private string $url;
    private string $verificationToken;
    private bool $logFail;
    private bool $logSuccess;
    private string $aid;
    private Parser $uaParser;

    private string $iframeUrl;
    private string $iframeNid;
    private string $iframeEventId;
    private string $iframeAdvEventId;

    private string $context = 'unknown';

    public static function init(): self
    {
        if (!self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        if (file_exists(__DIR__ . '/.env')) {
            $dotenv = Dotenv::createImmutable(__DIR__);
            $dotenv->load();
        }

        $this->uaParser = Parser::create();

        $this->url = $_ENV['EVERFLOW_URL'] ?? 'https://www.abjguytrk.com/sdk/conversion';
        $this->verificationToken = $_ENV['EVERFLOW_VERIFICATION_TOKEN'] ?? '8lhDp1R9TijaWUEssVIMzX5wZJKFTI';
        $this->aid = $_ENV['EVERFLOW_ADVERTISER_ID'] ?? '3';
        $this->logFail = !empty($_ENV['EVERFLOW_LOG_FAIL']) && $_ENV['EVERFLOW_LOG_FAIL'] === '1';
        $this->logSuccess = !empty($_ENV['EVERFLOW_LOG_SUCCESS']) && $_ENV['EVERFLOW_LOG_SUCCESS'] === '1';

        $this->iframeUrl = $_ENV['IFRAME_URL'] ?? 'https://www.abjguytrk.com/';
        $this->iframeNid = $_ENV['IFRAME_NID'] ?? '2380';
        $this->iframeEventId = $_ENV['IFRAME_EVENT_ID'] ?? '401';
        $this->iframeAdvEventId = $_ENV['IFRAME_ADV_EVENT_ID'] ?? '4';

        // Woo Action Scheduler
        add_action('everflow_process_order_async', [$this, 'handleAsyncPostback']);
        add_action('woocommerce_order_status_changed', [$this, 'schedulePostback'], 10, 4);

    }

    public function schedulePostback($orderId, $oldStatus, $newStatus, $order): void
    {
        if (!$order instanceof \WC_Order) {
            return;
        }

        $validOldStatuses = ['pending', 'on-hold', 'failed'];
        if (in_array($oldStatus, $validOldStatuses, true) && $newStatus === 'processing') {
            if (!as_has_scheduled_action('everflow_process_order_async', [$orderId], 'everflow')) {
                as_schedule_single_action(
                    time() + self::ASYNC_POSTBACK_DELAY,
                    'everflow_process_order_async',
                    [$orderId],
                    'everflow'
                );
            }
        }
    }

    private function runSDKRequest(\WC_Order $order, string $cid): void
    {
        $orderId = $order->get_id();
        $userAgent = $order->get_customer_user_agent();
        [$platform, $platformVersion] = $this->getOsPlatformAndVersion($userAgent);

        $amount = $order->get_total() - $order->get_shipping_total();
        $coupons = implode(',', $order->get_used_coupons());

        $itemsArr = [];
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            if ($product) {
                $itemsArr[] = [
                    'ps' => $product->get_sku() ?? 'unknown',
                    'qty' => $item->get_quantity(),
                    'p' => $order->get_line_total($item, true, true),
                ];
            }
        }

        $orderData = [
            'items' => $itemsArr,
            'oid' => $orderId,
            'amt' => sprintf('%.2f', $order->get_total()),
            'bs' => $order->get_billing_state(),
            'bc' => $order->get_billing_country(),
            'cc' => $coupons,
        ];

        $params = [
            'transaction_id' => $cid,
            'aid' => $this->aid,
            'amount' => $amount,
            'order_id' => $orderId,
            'verification_token' => $this->verificationToken,
            'email' => $order->get_billing_email(),
            'order' => wp_json_encode($orderData),

            'sec_ch_ua_platform' => $platform,
            'sec_ch_ua_platform_version' => $platformVersion,
            'sec_ch_ua_model' => '',
            'event_id' => 0,
            'coupon_code' => $coupons,
            'adv1' => '',
            'adv2' => '',
            'adv3' => '',
            'adv4' => '',
            'adv5' => '',
            'event_source_url' => $_SERVER['HTTP_HOST'] ?? '',
        ];
        [$success, $error] = $this->sendRequestWithoutRead($this->url, $params);
        $this->handleResponse($success, $orderId, $cid, $error);
    }

    public function runAllS2SRequests(\WC_Order $order): void
    {
        $orderId = $order->get_id();
        $cid = $this->getAnyStoredCID($order);
        if (empty($cid)) {
            $this->logSkipReason($order, 'No stored CID found');
            $order->update_meta_data(self::FIRE_STATUS_META_KEY, FireStatuses::NOCID);
            $order->save();
            return;
        }
        $this->runSDKRequest($order, $cid);
        $order->update_meta_data(self::FIRE_STATUS_META_KEY, FireStatuses::S2S);
        $order->save();

        $iframeUrls = $this->getAllIframeUrls($cid);
        foreach ($iframeUrls as $iframeUrl) {
            [$success, $error] = $this->sendRequestWithoutRead($iframeUrl, []);
            $this->handleResponse($success, $orderId, $cid, $error);
        }
    }

    private function getAnyStoredCID(\WC_Order $order): ?string
    {
        $actualCID = $order->get_meta(self::CID2_META_KEY);
        if (!empty($actualCID)) {
            return $actualCID;
        }

        $affiliateURL = $order->get_meta(self::AFFILIATE_URL_META_KEY);
        $affiliateCID = $this->getCIDFromURL($affiliateURL);
        return !empty($affiliateCID) ? $affiliateCID : null;
    }

    private function getOsPlatformAndVersion(string $userAgent): array
    {
        if (empty($userAgent)) {
            return ['Unknown', ''];
        }
        try {
            $result = $this->uaParser->parse($userAgent);

            $osFamily = $result->os->family ?: 'Unknown';
            $osVersion = $result->os->toVersion() ?: '';

            return [$osFamily, $osVersion];
        } catch (\Throwable $e) {
            $this->logToUploads('[UserAgent Parse Error] ' . $e->getMessage() . ' | UA: ' . $userAgent);
            return ['Unknown', ''];
        }
    }

    private function logToUploads(string $message): void
    {
        $uploads = wp_upload_dir();
        $logDir = $uploads['basedir'];
        $logFile = rtrim($logDir, '/\\') . '/everflow_debug.log';

        if (!is_dir($logDir) && !@mkdir($logDir, 0755, true)) {
            error_log('[Everflow] Failed to create log directory: ' . $logDir);
            return;
        }
        error_log($message . PHP_EOL, 3, $logFile);
    }

    private function getAllIframeUrls(string $cid): array
    {
        $baseParams = [
            [
                'nid' => $this->iframeNid,
                'verification_token' => $this->verificationToken,
                'transaction_id' => $cid,
            ],
            [
                'nid' => $this->iframeNid,
                'verification_token' => $this->verificationToken,
                'transaction_id' => $cid,
                'event_id' => $this->iframeEventId,
            ],
            [
                'nid' => $this->iframeNid,
                'verification_token' => $this->verificationToken,
                'transaction_id' => $cid,
                'adv_event_id' => $this->iframeAdvEventId,
            ]
        ];

        $urls = [];
        foreach ($baseParams as $params) {
            $urls[] = $this->iframeUrl . '?' . http_build_query($params);
        }
        return $urls;
    }

    private function sendRequestWithoutRead(string $url, array $params): array
    {
        $parts = parse_url($url);
        $host = $parts['host'] ?? '';
        $path = $parts['path'] ?? '/';
        $query = http_build_query($params);

        $fp = @fsockopen("ssl://{$host}", 443, $errno, $errstr, 1);

        if (!$fp) {
            return [false, sprintf('fsockopen failed (errno %d): %s', $errno, $errstr)];
        }

        $out = "GET {$path}?{$query} HTTP/1.1\r\n";
        $out .= "Host: {$host}\r\n";
        $out .= "Connection: Close\r\n\r\n";

        fwrite($fp, $out);
        stream_set_timeout($fp, 1, 0);
        fclose($fp);

        return [true, null];
    }

    private function logRequestResults(bool $success, int $orderId, string $cid, ?string $error = null): void
    {
        $status = $success ? 'SUCCESS' : 'FAIL';
        $message = sprintf(
            '[%s][Context: %s][Order #%d][CID: %s][%s]',
            date('Y-m-d H:i:s'),
            strtoupper($this->context),
            $orderId,
            $cid,
            $status
        );

        if (!$success && $error) {
            $message .= " Error: $error";
        }

        $this->logToUploads($message);
    }

    private function handleResponse(bool $success, int $orderId, string $cid, ?string $error = null): void
    {
        if ($success && $this->logSuccess) {
            $this->logRequestResults(true, $orderId, $cid, null);
        }

        if (!$success && $this->logFail) {
            $this->logRequestResults(false, $orderId, $cid, $error);
        }
    }

    private function getCIDFromURL(string $url): ?string
    {
        $parsedUrl = parse_url($url);
        if ($parsedUrl === false || !isset($parsedUrl['query'])) {
            return null;
        }

        parse_str($parsedUrl['query'], $queryParams);
        return isset($queryParams['cid']) ? sanitize_text_field($queryParams['cid']) : null;
    }

    public function handleAsyncPostback(int $orderId): void
    {
        $this->context = 's2s';
        $order = wc_get_order($orderId);
        if (!$order instanceof WC_Order) {
            return;
        }

        if ($this->isOrderFired($order)) {
            $this->logSkipReason($order, 'already fired');
            return;
        }

        $this->runAllS2SRequests($order);
    }

    private function isOrderFired(\WC_Order $order): bool
    {
        $fireStatus = $order->get_meta(self::FIRE_STATUS_META_KEY);
        $hasValue = !is_null($fireStatus);
        $notNoFireStatus = $fireStatus != FireStatuses::NO_FIRE;

        return $hasValue && $notNoFireStatus;
    }

    private function logSkipReason(\WC_Order $order, string $reason): void
    {
        $timestamp = date('Y-m-d H:i:s');
        $message = sprintf(
            '[%s][Order #%d] Skipped: %s. Log context %s',
            $timestamp,
            $order->get_id(),
            $reason,
            strtoupper($this->context)
        );

        $this->logToUploads($message);
    }

}

EverflowIntegration::init();
