<?php

declare( strict_types=1 );

namespace cloudflare_purge_cache\includes;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if (!class_exists('cloudflare_purge_cache\includes\Cloudflare_Purge_Cache_Core')) {
	class Cloudflare_Purge_Cache_Core {

		public function init() {
			add_action('admin_menu', [$this, 'init_submenu_page']);
			add_action('admin_bar_menu', [$this, 'init_admin_bar_menu'], 100);
		}

		public function init_submenu_page() {
			add_submenu_page(
				null,
				'Purge Cache',
				'Purge Cache',
				'manage_options',
				'purge-cloudflare-cache',
				[$this, 'purge_cache_callback']
			);

			add_submenu_page(
				null,
				'Purge Opcache',
				'Purge Opcache',
				'manage_options',
				'purge-opcache',
				[$this, 'purge_opcache_callback']
			);
		}

		public function init_admin_bar_menu($wp_admin_bar) {
			if (!is_admin() || !current_user_can('manage_options')) {
				return;
			}

			$wp_admin_bar->add_node([
				'id'    => 'cloudflare-menu',
				'title' => 'Cache Menu',
				'href'  => '#', // Prevent clicking
			]);

			$wp_admin_bar->add_node([
				'id'     => 'cloudflare-purge-cache',
				'title'  => 'Purge Cloudflare',
				'href'   => admin_url('admin.php?page=purge-cloudflare-cache'),
				'parent' => 'cloudflare-menu',
			]);

			$wp_admin_bar->add_node([
				'id'     => 'purge-opcache',
				'title'  => 'Purge Opcache',
				'href'   => admin_url('admin.php?page=purge-opcache'),
				'parent' => 'cloudflare-menu',
			]);
		}

		public function purge_cache_callback() {
			if (self::purge_cache()) {
				echo '<div class="updated"><p>Cloudflare cache purged successfully.</p></div>';
			} else {
				echo '<div class="error"><p>Some troubles when cache clear.</p></div>';
			}
		}

		public function purge_opcache_callback() {
			if (self::purge_opcache()) {
				echo '<div class="updated"><p>Opcache purged successfully.</p></div>';
			} else {
				echo '<div class="error"><p>Some troubles when opcache clear.</p></div>';
			}
		}

		static function purge_opcache(): bool {
			if (function_exists('opcache_reset')) {
				return opcache_reset();
			}

			return false;
		}

		static function purge_cache(): bool {
			$token = get_field('cloudflare_cache_token', 'option');

			if ( have_rows( 'cloudflare_cache_zones', 'option' ) ) {
				while ( have_rows( 'cloudflare_cache_zones', 'option' ) ) : the_row();
					$zone = get_sub_field('cloudflare_cache_zone');

					if (!self::send_purge_cache_request($token, $zone)) {
						return false;
					}
				endwhile;
			}

			return true;
		}

		static function send_purge_cache_request(string $token, string $zone): bool {
			if (!class_exists('WpOrg\Requests\Requests')) return false;

			$response = \WpOrg\Requests\Requests::post("https://api.cloudflare.com/client/v4/zones/$zone/purge_cache", [
				'Authorization' => "Bearer $token",
				'Content-Type' => 'application/json',
				'Accept'       => 'application/json',
			], json_encode(['purge_everything' => true]));

			$response = json_decode($response->body);

			return $response->success;
		}

	}
}