<?php

declare( strict_types=1 );

namespace checkout_reviews_update_dates\includes;

use WP_Query;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Checkout_Reviews_Update_Dates_Core {

	private static $instance = null;

	public static function get_instance(): self {
		if ( is_null( self::$instance ) ) {
			self::$instance = new static();
		}

		return self::$instance;
	}

	public function init(): void {
		add_action( 'admin_head', [ $this, 'add_button_to_views_posts_page' ] );

		add_action( 'admin_post_checkout_reviews_update_dates', [ $this, 'update_dates' ] );

		add_action( 'admin_notices', [ $this, 'show_admin_notices' ] );
	}

	public function update_dates(): void {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'У вас нет прав для выполнения этого действия.' );
		}

		$query = new WP_Query( [
			'post_type'      => 'checkout_review',
			'post_status'    => 'publish',
			'posts_per_page' => - 1
		] );

		$updated_posts = [];

		while ( $query->have_posts() ) {
			$query->the_post();

			$post_id = get_the_ID();

			$translated_post_id = apply_filters( 'wpml_object_id', $post_id, 'post', true );

			if ( in_array( $translated_post_id, $updated_posts ) ) {
				continue;
			}

			$updated_posts[] = $translated_post_id;

			$random_date = $this->randomDate( '-1 week', '-2 months 1 week' );

			$updated_post = array(
				'ID'            => $translated_post_id,
				'post_date'     => $random_date,
				'post_date_gmt' => get_gmt_from_date( $random_date ),
			);

			try {
				wp_update_post( $updated_post );
			} catch ( \Exception $_ ) {
			}
		}

		wp_safe_redirect( admin_url( 'edit.php?post_type=checkout_review&success=1' ) );
		exit;
	}

	public function add_button_to_views_posts_page(): void {
		$screen = get_current_screen();

		if ( $screen->id === 'edit-checkout_review' && current_user_can( 'manage_options' ) ) {
			$url = admin_url( 'admin-post.php?action=checkout_reviews_update_dates' );

			?>
            <script type="text/javascript">
                document.addEventListener("DOMContentLoaded", function () {
                    let addNewButton = document.querySelector('.page-title-action');
                    if (addNewButton) {
                        let myButton = document.createElement('a');
                        myButton.href = "<?php echo esc_url( $url ); ?>";
                        myButton.className = "page-title-action";
                        myButton.innerText = "Update Dates";

                        addNewButton.parentNode.insertBefore(myButton, addNewButton.nextSibling);
                    }
                });
            </script>
			<?php
		}
	}

	public function show_admin_notices(): void {
		if ( isset( $_GET['success'] ) && $_GET['success'] == '1' && ( isset( $_GET['post_type'] ) && $_GET['post_type'] == 'checkout_review' ) ) {
			echo '<div class="notice notice-success is-dismissible"><p>The update of the dates was successful!</p></div>';
		}
	}

	private function randomDate( $firstDate, $secondDate, $format = 'Y-m-d h:i:s' ): string {
		$firstDateTimeStamp  = strtotime( $firstDate, time() );
		$secondDateTimeStamp = strtotime( $secondDate, time() );

		if ( $firstDateTimeStamp < $secondDateTimeStamp ) {
			return date( $format, mt_rand( $firstDateTimeStamp, $secondDateTimeStamp ) );
		}

		return date( $format, mt_rand( $secondDateTimeStamp, $firstDateTimeStamp ) );
	}

}
