<?php

declare(strict_types=1);

namespace CrippleCrew\AdyenLite;

use WC_Payment_Gateway;
use WC_Order;
use WC_AJAX;

class Gateway extends WC_Payment_Gateway
{

    public function __construct()
    {
        $this->id = 'adyen_lite';
        $this->method_title = 'Adyen Lite';
        $this->method_description = 'Прямое подключение к Adyen без Woosa';
        $this->supports = ['refunds'];

        $this->init_form_fields();
        $this->init_settings();

        $this->enabled = $this->get_option('enabled');
        $this->title = $this->get_option('title');
        $this->test_mode = $this->get_option('test_mode', 'yes') === 'yes';
        $this->has_fields = true;

        add_action('wp_enqueue_scripts', [$this, 'enqueue_js']);
        // Веб‑хук Adyen → /?wc-api=adyen_lite
        add_action('woocommerce_api_' . $this->id, [$this, 'webhook_listener']);

        // Сохраняем опции, когда администратор их изменяет.
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
    }

    public function init_form_fields()
    {
        $this->form_fields = [
            'enabled'          => [
                'title'   => 'Включить',
                'type'    => 'checkbox',
                'label'   => 'Разрешить оплату картой',
                'default' => 'no'
            ],
            'test_mode'        => [
                'title'   => 'Test mode',
                'type'    => 'checkbox',
                'default' => 'yes'
            ],
            'title'            => [
                'title'   => 'Название на checkout',
                'type'    => 'text',
                'default' => 'Bank card (Adyen)'
            ],
            'merchant_account' => ['title' => 'Merchant account', 'type' => 'text'],
            'api_key'          => ['title' => 'API key', 'type' => 'password'],
            'client_key'       => ['title' => 'Client key', 'type' => 'text'],
            'hmac_key'         => ['title' => 'Webhook HMAC', 'type' => 'password'],
            'live_prefix'      => [
                'title'       => 'Live URL prefix',
                'type'        => 'text',
                'description' => 'Из Adyen CA → Developers → API URLs → Prefix (только буквы/цифры)',
                'default'     => '',
                'desc_tip'    => true,
            ],
        ];
    }

    /* ---------- checkout UI ---------- */
    /** called inside the radio-button when our method is selected */
    public function payment_fields()
    {
        echo '<p>Pay securely by card. You’ll be redirected for 3-D Secure if required.</p>';
        echo '<div id="adyen-card-container"></div>';
        echo '<input type="hidden" name="adyen_payment_data" id="adyen-payment-data" />';
    }

    /** enqueue Adyen JS + our helper */
    public function enqueue_js(): void
    {
        if (!is_checkout()) {
            return;
        }

        // Adyen SDK (pin version you use in test)
        wp_enqueue_script(
            'adyen-sdk',
            'https://checkoutshopper-live.adyen.com/checkoutshopper/sdk/6.12.0/adyen.js',
            [],
            null,
            true
        );

        wp_enqueue_style(
            'adyen-sdk',
            'https://checkoutshopper-live.adyen.com/checkoutshopper/sdk/6.12.0/adyen.css',
            [],
            null
        );

        // our integrator – very small
        wp_enqueue_script(
            'adyen-lite',
            plugins_url('assets/adyen-lite.js', dirname(__FILE__, 2) . '/adyen-lite.php'),
            ['jquery', 'adyen-sdk'],
            '1.0',
            true
        );

        // pass run-time data
        wp_localize_script(
            'adyen-lite',
            'wc_adyen_lite',
            [
                // сразу выдаём Полный URL именно к нашему энд-пойнту
                'session_url' => WC_AJAX::get_endpoint('adyen_lite_session'),
                'clientKey'   => $this->get_option('client_key'),
                'gateway_id'  => $this->id,
            ]
        );
    }

    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);

        dd($order, $_POST);
        $order->update_status('pending', 'Waiting for Adyen authorisation');

        // classic checkout flow: redirect to order-received page
        return [
            'result'   => 'success',
            'redirect' => $this->get_return_url($order),
        ];
    }

    private function handle_adyen_response(array $result, WC_Order $order): array
    {
        // Сохраним pspReference, если уже пришёл
        if (!empty($result['id'])) {
            $order->update_meta_data('_adyen_session_id', $result['id']);
            $order->save();
        }

        // Drop-in нужен redirectUrl и sessionData
        if (isset($result['url'])) {
            return [
                'result'   => 'success',
                'redirect' => $result['url'],
            ];
        }

        wc_add_notice('Не удалось создать платёжную сессию.', 'error');
        return ['result' => 'failure'];
    }

    public function webhook_listener(): void
    {
        $raw = file_get_contents('php://input');
        $items = json_decode($raw, true)['notificationItems'] ?? [];

        $handler = new WebhookHandler($this->settings['hmac_key']);

        foreach ($items as $it) {
            $handler->handle($it['NotificationRequestItem']);
        }

        // Требуемый ответ Adyen: {"notificationResponse":"[accepted]"}
        wp_send_json(['notificationResponse' => '[accepted]']);
    }
}
