<?php

declare(strict_types=1);

namespace CrippleCrew\AdyenLite;

use Adyen\AdyenException;
use Adyen\Client;
use Adyen\Environment;
use Adyen\Service\Checkout\PaymentsApi;
use Adyen\Service\Checkout\ModificationsApi;
use Adyen\Model\Checkout\CreateCheckoutSessionRequest;
use Adyen\Model\Checkout\Amount;
use Adyen\Model\Checkout\PaymentRefundRequest;
use WC_Order;

class AdyenLiteService
{
    private PaymentsApi $payments;
    private ModificationsApi $mods;
    private string $merchant;

    /**
     * @throws AdyenException
     * @var array{"api_key":string, "test_mode":boolean, "live_prefix":null|string,"merchant_account":null|string} $cfg
     */
    public function __construct(array $cfg)
    {
        $client = new Client();
        $client->setXApiKey($cfg['api_key']);
        $client->setEnvironment(
            $cfg['test_mode'] === true ? Environment::TEST : Environment::LIVE,
            $cfg['live_prefix'] ?? null
        );

        $this->payments = new PaymentsApi($client);       // новый сервис :contentReference[oaicite:0]{index=0}
//        $this->mods = new ModificationsApi($client);  // для refund'ов
        $this->merchant = $cfg['merchant_account'];
    }


    public function createSessionFromCart(string $reference): \stdClass
    {
        $total = WC()->cart->get_total('edit');
        $currency = get_woocommerce_currency();
        $country  = WC()->customer->get_billing_country();

        $req = (new CreateCheckoutSessionRequest())
            ->setMerchantAccount($this->merchant)
            ->setReference($reference)
            ->setAmount(
                (new Amount())
                    ->setCurrency($currency)
                    ->setValue((int)round($total * 100))
            )
            ->setReturnUrl(home_url('/thank-you'))
            ->setChannel( CreateCheckoutSessionRequest::CHANNEL_WEB )
            ->setCountryCode( $country );

        return $this->payments->sessions($req)->jsonSerialize();
    }

    public function makePayment(int $orderId, array $pmData): array {
        $order = wc_get_order($orderId);
        $request = [
            'merchantAccount'   => $this->merchant,
            'reference'         => (string)$orderId,
            'amount'            => ['currency'=>$order->get_currency(),'value'=> (int)round($order->get_total()*100)],
            'paymentMethod'     => $pmData,
            'returnUrl'         => $this->returnUrl($order),
            'shopperReference'  => (string)$order->get_customer_id(),
        ];
        $response = $this->payments->payments($request);
        return $response->jsonSerialize();
    }

}
